/*
 * Nid d'abeille
 *
 * Auteur original:
 * fumbucker (Xiao Jin) @ thingiverse
 * https://www.thingiverse.com/thing:2180174
 * Licence: 
 * https://creativecommons.org/licenses/by/4.0/
 *
 * Modifications: Christophe Boyanique
 * Licence: CC Attribution-Share Alike 4.0 International
 * https://creativecommons.org/licenses/by-sa/4.0/deed.fr
 *
 */
 
/*
Honeycomb generator module
*/
/*
module honeycomb generate honeycomb based on number of rows and columns
  r float radius of hexagons
  thickness float thickness of hexagon walls
  cols int how many hexagons per row
  rows int number of rows
  center bool center generated shape
  
  example:
    color("Yellow") honeycomb(r=4, thickness=2, cols=5, rows=4, center=true);
*/
module honeycomb(r, thickness=1, cols=0, rows=0, center=false, inverse=false) {
  translate([ // if center is true, calculate total size and translate accordingly
    center ? ( cols * r / tan(30) + (rows==1?-1:1)*(r + thickness) / tan(30) / 2 ) /-2 : 0,
    center ? ( rows * r * sin(30) + (rows+0.5) * r + thickness ) /-2 : 0
  ])
  translate([ (rows==1?0.33:1)*(r+thickness/2) / 2 / tan(30), r+thickness/2])  // Move so it starts at 0, 0
  for (j = [ 0 : 1 : rows-1 ] ) {  // Iterate rows
    translate([ r / 2 / tan(30) * (j%2) , j * (r + r * sin(30)) ])
    for (i = [ 0 : 1 : cols-1 ]) { // Iterate columns
      translate([ i * ( r / tan(30) ) , 0]) rotate([0, 0, 90]) {

        // Make the hexagons
        if (thickness==0) {
          circle(r, $fn=6);
        }
        else {
          if (inverse) {
              circle(r-thickness/2, $fn=6);
          } else {
            difference() {
              circle(r+thickness/2, $fn=6);
              circle(r-thickness/2, $fn=6);
            }
          }
        }
      }
    }
  }
}

module honeycomb_cyl(r, thickness=1, cols=0, rows=0, h=1, center=false, inverse=false) {
  translate([ // if center is true, calculate total size and translate accordingly
    center ? ( cols * r / tan(30) + (rows==1?-1:1)*(r + thickness) / tan(30) / 2 ) /-2 : 0,
    center ? ( rows * r * sin(30) + (rows+0.5) * r + thickness ) /-2 : 0
  ])
  translate([ /*(rows==1?0.33:1)*/1*(r+thickness/2) / 2 / tan(30), r+thickness/2])  // Move so it starts at 0, 0
  for (j = [ 0 : 1 : rows-1 ] ) {  // Iterate rows
    translate([ r / 2 / tan(30) * (j%2) , j * (r + r * sin(30)) ])
    for (i = [ 0 : 1 : cols-1 ]) { // Iterate columns
      translate([ i * ( r / tan(30) ) , 0]) rotate([0, 0, 90]) {

        // Make the hexagons
        if (thickness==0) {
          cylinder(r=r, $fn=6, h=h);
        }
        else {
          if (inverse) {
              cylinder(r=r-thickness/2, $fn=6, h=h);
          } else {
            difference() {
              cylinder(r=r+thickness/2, $fn=6, h=h);
              cylinder(r=r-thickness/2, $fn=6, h=h);
            }
          }
        }
      }
    }
  }
}

  
  /* // Draw a square around the total area
  %square([
    cols * r / tan(30) + (r + thickness) / tan(30) / 2,
    rows * r * sin(30) + (rows+0.5) * r + thickness
  ]);
  */

/*
module honeycomb_square generate honeycomb based on x and y size
  dimensions vector [x, y] size of square
  r float radius of hexagons
  thickness float thickness of hexagon walls
  center bool center generated shape
  
  example:
    color("Yellow") honeycomb_square([100, 20], 2, 1, true);
*/
module honeycomb_square(dimensions = [10, 10], r=1, thickness=1, center=false) {
  
  intersection() {
    translate([r * -1, thickness * -1])
    honeycomb(r=r, thickness=thickness,
      cols = (dimensions[0] - (r + thickness) / tan(30) / 2) / (r / tan(30)) + 2,
      rows = dimensions[1] / (r*2 - thickness) + 1,
      center=center);
   
    square(dimensions, center=center); 
  }
}

function honeycomb_calc_ncols(r, thickness, w) = floor((w - (r + thickness) / tan(30) / 2) * tan(30) / r);

function honeycomb_calc_nrows(r, thickness, h) = floor((h - 0.5*r - thickness) / (r*sin(30) + r));

function honeycomb_calc_w(r, thickness, cols, rows) = cols * r / tan(30) + (r + thickness) / tan(30) / 2 - (rows==1?1:0)*(r-thickness/2);

function honeycomb_calc_h(r, thickness, cols, rows) = rows * r * sin(30) + (rows+0.5) * r + thickness;

function honeycomb_calc_x(r, thickness, col, row) = 
    (r+thickness/2) / 2 / tan(30) + 
    r / 2 / tan(30) * (row%2) +
    col * ( r / tan(30) );

function honeycomb_calc_y(r, thickness, col, row) = 
    r+thickness/2 +
    row * (r + r * sin(30));

// https://fr.wikihow.com/calculer-l%27apoth%C3%A8me-d%27un-hexagone
function honeycomb_calc_apo(r, thickness) = (r-thickness/2) / (2*tan(30));

/*
nx = 1;
ny = 15;
x  = 0;
y  = 10;

                //rotate([0, 90, 0])
honeycomb_cyl(r=4, thickness=1, cols=nx, rows=ny, h=1, center=false, inverse=true);

translate([
    honeycomb_calc_x(4, 1, x, y),
    honeycomb_calc_y(4, 1, x, y),
    1
])
                //rotate([0, 90, 0])
    color("red", alpha=0.25)
        circle(r=3.5, $fn=32);

translate([
    honeycomb_calc_x(4, 1, x, y) - honeycomb_calc_apo(4, 1),
    -25,
    ,0
])
                //rotate([0, 90, 0])
    color("blue", alpha=0.25)
        cube([0.1, 50, 1]);

translate([0, 0, -1])
                //rotate([0, 90, 0])
    color("skyblue")
        cube([honeycomb_calc_w(4, 1, nx, ny), honeycomb_calc_h(4, 1, nx, ny), 1]);
*/