
/*
 * Boîtier détecteur CO2 v3
 *
 * Auteur: Christophe Boyanique
 * Licence: CC-BY-NC (CC Attribution-Share Alike 4.0 International)
 * https://creativecommons.org/licenses/by-sa/4.0/deed.fr
 *
 * Exceptions (voir en-têtes):
 *  rcm_honeycomb.lib  fumbucker@thingiverse CC-BY
 *  rcm_plq.lib          vland70@thingiverse CC-BY-NC
 *  rcm_dupont.lib       vland70@thingiverse CC-BY-NC
 *
 */

include <../lib/rcm_ky040.scad>;
include <../lib/rcm_s16340.scad>;
include <../lib/rcm_plq.scad>;
include <../lib/rcm_epd.scad>;
include <../lib/rcm_dupont.scad>;
include <../lib/rcm_s8.scad>;
include <../lib/rcm_d32pro.scad>;
include <../lib/rcm_honeycomb.scad>;

vide           = 0.01;
buse_w         = 0.4;
buse_h         = 0.2;
res            = 32;
prt            = 1;
pente          = 3;

o_microsd   = 1;

o_box       = 1;
o_top       = 1;
o_plq_epd   = 1;
o_knob1     = 1;
o_knob2     = 1;

o_bx_bottom = 1;
o_bx_front  = 1;
o_bx_left   = 1;
o_bx_back   = 1;
o_bx_right  = 1;

draw_ky040  = 1;
draw_s16340 = 1;
draw_epd    = 1;
draw_plq    = 1;
draw_s8     = 1;
draw_d32    = 1;

dupontf_picot_h = 3;
dupontf_body_h = 8.4;
dupontm_picot_h = 3;
dupontm_body_h = 2.4;
dupontm_male_h = 6;

bx_ep      =   2.0;
bx_ep_fin  =   1.2;
letter_depth = 0.8;

event_bee_r    = 4.8;
event_bee_wall = 2.4;

o_slogan = 0;
font = "Quicksand:style=Bold";
letter_size = 8;
slogan = "Le 17";

sw_letter_depth = 0.8;
sw_letter_size  = 4;

bx_r = bx_ep / 2;
ep_ky040_s16340 = 2;
cale            =  1.6;

led_h          =  7.6;
led_r          =  2.6;

s16340_corner_r = 2.2;

plq_l = 70.4;
plq_w = 30.2;
plq_h =  1.6;
plq_nx = 24;
plq_ny = 10;
plq_yoff = 1*bx_ep; // Espace entre le PCB et le fond

epd_y = bx_ep -2*buse_w*0 + 0.5*buse_w;
epd_z = bx_ep + _epd_tft_b + 0.5*buse_w;

plq_y = epd_y + _epd_scr_t + _epd_pcb_t + _epd_tft_h - 0.0*buse_w;

s16340_y = plq_y + (plq_w - s16340_w) + 1.5*buse_w + plq_yoff;
s16340_z = bx_ep + s16340_b + 0.5*buse_w;
plq_z = s16340_z + s16340_height() + buse_w + bx_ep + cale;

bx_l = bx_ep + cale/2
    + _s8_l - rcm_s8h5_x(0)
    + rcm_plq_x(plq_l, plq_nx, 4) - rcm_plq_x(plq_l, plq_nx, 0)
    + _d32pro_l - rcm_d32pro_hb_x(0)
    + cale/2 + bx_ep;
bx_w = plq_y + plq_w + 0.5*buse_w
    + bx_ep  // Espace derrière l'écran
    + plq_yoff; // Espace entre le pcb et le fond
bx_h = plq_z + plq_h + dupontf_picot_h + dupontf_body_h + dupontm_body_h + 2*cale + 2*bx_ep;

rotor_r = 15.0/2;
rotor_h = 17.0;

plq_x = bx_l - bx_ep
    - _d32pro_l + rcm_d32pro_hb_x(0)
    - rcm_plq_x(plq_l, plq_nx, 4)
    - cale/2;

epd_x = plq_x;
s16340_x = bx_l - bx_ep - s16340_l - 0.5*buse_w;

ky040_x = plq_x - 0.5*buse_w -cale + _ky040_h + _ky040base_h + _ky040axe_h - _ky040ecrou_h - 0.5*buse_w
- _ky040axe_h+_ky040ecrou_h+buse_w + cale ;
ky040_y = s16340_y-0.5*buse_w+2*s16340_corner_r+0.5*buse_w;
ky040_z = bx_w - ky040_y- _ky040axe_y - _ky040axe_y;

d32pro_x = plq_x + rcm_plq_x(plq_l, plq_nx, 4) - rcm_d32pro_hb_x(0);
d32pro_y = plq_y + rcm_plq_y(plq_w, plq_ny, 0) - rcm_d32pro_hb_y(0);
d32pro_z = plq_z + plq_h + dupontf_body_h + dupontm_body_h;
s8_x = plq_x + rcm_plq_x(plq_l, plq_nx, 0) + rcm_s8h5_x(0);
s8_y = plq_y + rcm_plq_y(plq_w, plq_ny, 0) + rcm_s8h5_y(0);
s8_z = plq_z + plq_h + dupontf_body_h + dupontm_body_h;

// Renfoncement port µUSB+leds
renf_r = (bx_ep-bx_ep_fin);
renf_x = bx_l;
renf_y = s16340_y+s16340leds_y-1.5*bx_ep;
renf_w = s16340_w/2+s16340musb_w/2-s16340leds_y+3*bx_ep;
renf_z = s16340_z+s16340_h-1.5*bx_ep;
renf_h = s16340musb_h+3.0*bx_ep;

led_x = (epd_x + _epd_disp_x1) / 2;
led_z = epd_z + _epd_scr_y + _epd_scr_w/3.25;

echo(bx_l, bx_w, bx_h);

nevtfrthx = 2;
nevtfrthy = 5;
nevtfrthw = honeycomb_calc_w(event_bee_r, event_bee_wall, nevtfrthx, nevtfrthy);
nevtfrthh = honeycomb_calc_h(event_bee_r, event_bee_wall, nevtfrthx, nevtfrthy);

nevtrgthx = 2;
nevtrgthy = 3;
nevtrgthw = honeycomb_calc_w(event_bee_r, event_bee_wall, nevtrgthx, nevtrgthy);
nevtrgthh = honeycomb_calc_h(event_bee_r, event_bee_wall, nevtrgthx, nevtrgthy);

nevtbckhx = 1;
nevtbckhy = 13;
nevtbckhw = honeycomb_calc_w(event_bee_r, event_bee_wall, nevtbckhx, nevtbckhy);
nevtbckhh = honeycomb_calc_h(event_bee_r, event_bee_wall, nevtbckhx, nevtbckhy);

nevtlfthx = 2;
nevtlfthy = 3;
nevtlfthw = honeycomb_calc_w(event_bee_r, event_bee_wall, nevtlfthy, nevtlfthx);
nevtlfthh = honeycomb_calc_h(event_bee_r, event_bee_wall, nevtlfthx, nevtlfthy);

plq_epd_l = _epd_pcb_l;
plq_epd_w = bx_h - bx_ep - epd_z - _epd_pcb_w - 0*buse_w/2;
plq_epd_t = _epd_pcb_t;
plq_epd_hole_x = d32pro_x+_d32pro_l-_d32prosd_x-_d32prosd_l-buse_w-epd_x;
plq_epd_hole_y = bx_h-d32pro_z-_d32prosd_h+buse_w/2;
plq_epd_hole_l = _d32prosd_l+2*buse_w;
plq_epd_hole_w = _d32prosd_h+2*buse_w;

knob1_r = 6.0;
knob1_h =14.8;
knob2_h = 5.6;
knob2_r = 1.2;

module boite() {

    if (o_bx_bottom) {
        fond();
    }

    if (o_bx_front) {
        // Devant:
        difference() {
            union() {
                hull() {
                    translate([bx_r, bx_r, bx_ep])
                        cylinder(r=bx_r, h=bx_h-2*bx_ep, $fn=res);
                    translate([bx_l-bx_r, bx_r, bx_ep])
                        cylinder(r=bx_r, h=bx_h-2*bx_ep, $fn=res);
                }
                // cheminée leds
                hull() {
                    translate([led_x, bx_ep, led_z])
                        rotate([-90, 0, 0])
                            cylinder(r=led_r+bx_ep_fin, h=led_h-bx_ep, $fn=res);
                    translate([led_x, 0, max(bx_ep+led_r+bx_ep_fin, led_z-pente*(led_h-bx_ep))])
                        rotate([-90, 0, 0])
                            cylinder(r=led_r+bx_ep_fin, h=bx_ep, $fn=res);
                }
            }
            // Trou led
            translate([led_x, -vide, led_z])
                rotate([-90, 0, 0])
                    cylinder(r=led_r, h=led_h+2*vide, $fn=res);

            // Écran devant
            hull() {
                translate([epd_x+_epd_disp_x1+2, bx_ep+vide, epd_z+_epd_disp_y1+2])
                    rotate([90, 0, 0])
                        cylinder(r=2, h=bx_ep+2*vide, $fn=res);
                translate([epd_x+_epd_disp_x1+_epd_disp_l-2, bx_ep+vide, epd_z+_epd_disp_y1+2])
                    rotate([90, 0, 0])
                        cylinder(r=2, h=bx_ep+2*vide, $fn=res);
                translate([epd_x+_epd_disp_x1+_epd_disp_l-2, bx_ep+vide, epd_z+_epd_disp_y1+_epd_disp_w-2])
                    rotate([90, 0, 0])
                        cylinder(r=2, h=bx_ep+2*vide, $fn=res);
                translate([epd_x+_epd_disp_x1+2, bx_ep+vide, epd_z+_epd_disp_y1+_epd_disp_w-2])
                    rotate([90, 0, 0])
                        cylinder(r=2, h=bx_ep+2*vide, $fn=res);
            }

            // MicroSD
            if (o_microsd) {
                translate([d32pro_x+_d32pro_l-_d32prosd_x-_d32prosd_l-buse_w, -vide, d32pro_z-_d32prosd_h-buse_w])
                    cube([_d32prosd_l+2*buse_w, bx_ep+2*vide, _d32prosd_h+2*buse_w]);
            }

            // Aération
        translate([led_x - nevtfrthh/2 + 2*buse_w, bx_ep--vide, bx_h - 2*bx_ep])
            //color("magenta")
            rotate([90, 90, 0])
                honeycomb_cyl(r=event_bee_r, thickness=event_bee_wall, cols=nevtfrthx, rows=nevtfrthy-2, h=bx_ep+2*vide, center=false, inverse=true);

            if (o_slogan) {
                translate([(epd_x+_epd_disp_x1)/2, +letter_depth, epd_z + _epd_scr_y + _epd_scr_w*1/2+letter_size/2])
                    rotate([90, 0, 0])
                    //color("DodgerBlue")
                        linear_extrude(height = letter_depth+vide) {

                            text(slogan, size = letter_size, font = font, halign = "center", valign = "center", $fn = res);
                        }
            }
        }
    }

    if (o_bx_right) {
        // Côté droit
        difference() {
            hull() {
                translate([bx_l-bx_r, bx_r, bx_ep])
                    cylinder(r=bx_r, h=bx_h-2*bx_ep, $fn=res);
                translate([bx_l-bx_r, bx_w-bx_r, bx_ep])
                    cylinder(r=bx_r, h=bx_h-2*bx_ep, $fn=res);
            }

            // port µUSB+led
            hull() {
                translate([renf_x, renf_y, renf_z])
                    sphere(r=renf_r, $fn=res);
                translate([renf_x, renf_y+renf_w, renf_z])
                    sphere(r=renf_r, $fn=res);
                translate([renf_x, renf_y+renf_w, renf_z+renf_h])
                    sphere(r=renf_r, $fn=res);
                translate([renf_x, renf_y, renf_z+renf_h])
                    sphere(r=renf_r, $fn=res);
            }
            // port µUSB
            hull() {
                translate([bx_l-bx_ep-vide, s16340_y+s16340_w/2-s16340musb_w/2-buse_w/2+0.2, s16340_z+s16340_h+0.2])
                    rotate([0, 90, 0])
                        cylinder(r=0.2, h=bx_ep, $fn=res);
                translate([bx_l-bx_ep-vide, s16340_y+s16340_w/2+s16340musb_w/2-buse_w/2-0.2, s16340_z+s16340_h+0.2])
                    rotate([0, 90, 0])
                        cylinder(r=0.2, h=bx_ep, $fn=res);
                translate([bx_l-bx_ep-vide, s16340_y+s16340_w/2+s16340musb_w/2-buse_w/2-0.2, s16340_z+s16340_h+s16340musb_h-0.2])
                    rotate([0, 90, 0])
                        cylinder(r=0.2, h=bx_ep, $fn=res);
                translate([bx_l-bx_ep-vide, s16340_y+s16340_w/2-s16340musb_w/2-buse_w/2+0.2, s16340_z+s16340_h+s16340musb_h-0.2])
                    rotate([0, 90, 0])
                        cylinder(r=0.2, h=bx_ep, $fn=res);
            }

            // Leds
            translate([bx_l-bx_ep-vide, s16340_y+s16340leds_y+s16340leds_w/2, s16340_z+s16340_h+s16340leds_h])
                rotate([0, 90, 0])
                    cylinder(r=s16340leds_h, h=bx_ep+2*vide, $fn=res);

            // Aération
            translate([bx_l-2*bx_ep,  (bx_w-nevtrgthh)/2, bx_h-2*bx_ep])
                //color("magenta")
                rotate([0, 90, 0])
                        honeycomb_cyl(r=event_bee_r, thickness=event_bee_wall, cols=nevtrgthx, h=3*bx_ep, rows=nevtrgthy, center=false, inverse=true);
        }
    }

    if (o_bx_back) {
        // Fond
        difference() {
            hull() {
                translate([bx_r, bx_w-bx_r, bx_ep])
                    cylinder(r=bx_r, h=bx_h-2*bx_ep, $fn=res);
                translate([bx_l-bx_r, bx_w-bx_r, bx_ep])
                    cylinder(r=bx_r, h=bx_h-2*bx_ep, $fn=res);
            }
            // interrupteur
            hull() {
                translate([s16340_x+s16340sw_x+0.2, bx_w+vide, s16340_z+s16340_h+buse_h+0.2])
                    rotate([90, 0, 0])
                        cylinder(r=0.2, h=bx_ep+2*vide, $fn=res);
                translate([s16340_x+s16340sw_x+s16340sw_l-0.2, bx_w+vide, s16340_z+s16340_h+buse_h+0.2])
                    rotate([90, 0, 0])
                        cylinder(r=0.2, h=bx_ep+2*vide, $fn=res);
                translate([s16340_x+s16340sw_x+s16340sw_l-0.2, bx_w+vide, s16340_z+s16340_h+s16340sw_h+buse_h-0.2])
                    rotate([90, 0, 0])
                        cylinder(r=0.2, h=bx_ep+2*vide, $fn=res);
                translate([s16340_x+s16340sw_x+0.2, bx_w+vide, s16340_z+s16340_h+s16340sw_h+buse_h-0.2])
                    rotate([90, 0, 0])
                        cylinder(r=0.2, h=bx_ep+2*vide, $fn=res);
            }

        // Texte interrupteur
        translate([s16340_x+s16340sw_x - sw_letter_size*0.66, bx_w - sw_letter_depth, s16340_z + s16340_h + buse_h + s16340sw_h/2])
            rotate([90, 0, 180])
            //color("DodgerBlue")
                linear_extrude(height = sw_letter_depth+vide) {
                    text("0", size = sw_letter_size, font = font, halign = "center", valign = "center", $fn = res);
                        }
        translate([s16340_x+s16340sw_x + s16340sw_l + sw_letter_size/2, bx_w - sw_letter_depth, s16340_z + s16340_h + buse_h + s16340sw_h/2])
            rotate([90, 0, 180])
            //color("DodgerBlue")
                linear_extrude(height = sw_letter_depth+vide) {
                    text("1", size = sw_letter_size, font = font, halign = "center", valign = "center", $fn = res);
                        }


        // Aération
        translate([(bx_l-nevtbckhh)/2, bx_w+vide, bx_h - 2*bx_ep])
            //color("magenta")
            rotate([90, 90, 0])
                honeycomb_cyl(r=event_bee_r, thickness=event_bee_wall, cols=nevtbckhx, rows=nevtbckhy, h=bx_ep+2*vide, center=false, inverse=true);
        }
    }

    if (o_bx_left) {
        // Côté gauche
        difference() {
            hull() {
                translate([bx_r, bx_w-bx_r, bx_ep])
                    cylinder(r=bx_r, h=bx_h-2*bx_ep, $fn=res);
                translate([bx_r, bx_r, bx_ep])
                    cylinder(r=bx_r, h=bx_h-2*bx_ep, $fn=res);
            }
        // trou sphere
        translate([0, ky040_y+_ky040axe_y, ky040_z+_ky040axe_x])
            sphere(r=ky040_z+_ky040axe_x, $fn=res);
        // Aération
        translate([-bx_ep,  (bx_w-nevtlfthh)/2, bx_h-2*bx_ep])
            //color("magenta")
            rotate([0, 90, 0])
                honeycomb_cyl(r=event_bee_r, thickness=event_bee_wall, cols=nevtlfthx, h=3*bx_ep, rows=nevtlfthy, center=false, inverse=true);
        }

    difference() {
        // coque
        translate([0, ky040_y+_ky040axe_y, ky040_z+_ky040axe_x])
            sphere(r=ky040_z+_ky040axe_x, $fn=res);
        translate([0, ky040_y+_ky040axe_y, ky040_z+_ky040axe_x])
            sphere(r=ky040_z+_ky040axe_x-bx_ep_fin, $fn=res);
        // extérieur
        translate([-ky040_z-_ky040axe_x-vide, 0, 0])
            cube([ky040_z+_ky040axe_x+vide, bx_w, bx_h]);
        // tunnel
        translate([0, ky040_y+_ky040axe_y, ky040_z+_ky040axe_x])
            rotate([0, 90, 0])
                cylinder(r=rotor_r+2*buse_w, h=ky040_z+_ky040axe_x+vide, $fn=res);
    }
}

    // Sous s16340
    // plot devant/gauche
    hull() {
        translate([s16340_x+s16340_corner_r-0.5*buse_w, s16340_y+s16340_corner_r-0.5*buse_w, bx_ep])
            cylinder(r=s16340_corner_r, h=s16340_b, $fn=res);
        translate([s16340_x-0.5*buse_w, s16340_y-cale-0.5*buse_w, bx_ep])
            cube([2*s16340_corner_r, cale, s16340_b]);
        translate([s16340_x-0.5*buse_w-cale, s16340_y-0.5*buse_w, bx_ep])
            cube([cale, 2*s16340_corner_r, s16340_b]);
        translate([s16340_x-cale/2-0.5*buse_w, s16340_y-cale/2-0.5*buse_w, bx_ep])
            cylinder(r=cale/2, h=s16340_b, $fn=res);
    }
    hull() {
        translate([s16340_x-0.5*buse_w, s16340_y-cale-0.5*buse_w, bx_ep+s16340_b])
            cube([2*s16340_corner_r, cale, s16340_h+buse_w]);
        translate([s16340_x-cale/2-0.5*buse_w, s16340_y-cale/2-0.5*buse_w, bx_ep+s16340_b])
            cylinder(r=cale/2, h=s16340_h+buse_w, $fn=res);
    }
    hull() {
        translate([s16340_x-0.5*buse_w-cale, s16340_y-0.5*buse_w, bx_ep+s16340_b])
            cube([cale, 2*s16340_corner_r, s16340_h+buse_w]);
        translate([s16340_x-cale/2-0.5*buse_w, s16340_y-cale/2-0.5*buse_w, bx_ep+s16340_b])
            cylinder(r=cale/2, h=s16340_h+buse_w, $fn=res);
    }
    // picôt
    translate([s16340_x+s16340tl_x, s16340_y+s16340tl_y, bx_ep+s16340_b])
            cylinder(r=s16340t_r-1.5*buse_w, h=s16340_h, $fn=res);

    // plot devant/droite
    hull() {
        translate([bx_l-2*bx_ep, s16340_y-0.5*buse_w-buse_w/2, bx_ep])
            cube([bx_ep, 2*s16340_corner_r, s16340_b]);
        translate([s16340_x+s16340_l-0.5*buse_w-s16340_corner_r, s16340_y-0.5*buse_w-buse_w/2+s16340_corner_r, bx_ep])
            cylinder(r=s16340_corner_r, h=s16340_b, $fn=res);
        translate([s16340_x+s16340_l-0.5*buse_w-2*s16340_corner_r, s16340_y-0.5*buse_w-buse_w/2-cale, bx_ep])
            cube([2*s16340_corner_r+buse_w, cale, s16340_b]);
    }
    translate([s16340_x+s16340_l-0.5*buse_w-2*s16340_corner_r, s16340_y-0.5*buse_w-buse_w/2-cale, bx_ep+s16340_b])
            cube([2*s16340_corner_r+buse_w, cale, s16340_h+buse_w]);
    // picôt
    translate([s16340_x+s16340_l-s16340tr_x, s16340_y+s16340tl_y, bx_ep+s16340_b])
            cylinder(r=s16340t_r-1.5*buse_w, h=s16340_h, $fn=res);

    // plot fond/droite
    hull() {
        translate([bx_l-2*bx_ep, s16340_y+s16340_w-2*s16340_corner_r, bx_ep])
            cube([bx_ep, 2*s16340_corner_r, s16340_b]);
        translate([s16340_x+s16340_l-2*s16340_corner_r, bx_w-2*bx_ep, bx_ep])
            cube([2*s16340_corner_r+buse_w/2, bx_ep, s16340_b]);
        translate([s16340_x+s16340_l-s16340_corner_r, s16340_y+s16340_w-s16340_corner_r, bx_ep])
            cylinder(r=s16340_corner_r, h=s16340_b, $fn=res);
    }
    // picôt
   /*translate([s16340_x+s16340_l-s16340tr_x, s16340_y+s16340_w-s16340tl_y, bx_ep+s16340_b])
            cylinder(r=s16340t_r-buse_w, h=s16340_h, $fn=res);*/

    // plot fond/gauche
    hull() {
        translate([s16340_x-0.5*buse_w, bx_w-bx_ep, bx_ep])
            cube([2*s16340_corner_r, bx_ep, s16340_b]);
        translate([s16340_x-0.5*buse_w-cale, s16340_y+s16340_w-2*s16340_corner_r, bx_ep])
            cube([cale, bx_ep+2*s16340_corner_r-buse_w, s16340_b]);
        translate([s16340_x-0.5*buse_w+s16340_corner_r, s16340_y+s16340_w-s16340_corner_r, bx_ep])
            cylinder(r=s16340_corner_r, h=s16340_b, $fn=res);
    }
        translate([s16340_x-0.5*buse_w-cale, s16340_y+s16340_w-2*s16340_corner_r, bx_ep])
            cube([cale, bx_ep+2*s16340_corner_r-buse_w, s16340_b+s16340_h+buse_w]);
    // picôt
    /*translate([s16340_x+s16340tl_x, s16340_y+s16340_w-s16340tl_y, bx_ep+s16340_b])
            cylinder(r=s16340t_r-buse_w, h=s16340_h, $fn=res);*/

    // ePD à droite
    // dessous
    hull() {
        translate([epd_x + _epd_pcb_l - cale, epd_y + _epd_scr_t-0.5*buse_w-cale, bx_ep])
            cube([cale+0.5*buse_w, cale, _epd_tft_b]);

        translate([epd_x + _epd_pcb_l - cale, epd_y + _epd_scr_t+_epd_pcb_t+0.5*buse_w, bx_ep])
            cube([cale+0.5*buse_w, cale, _epd_tft_b]);

    }
    // devant
    translate([epd_x + _epd_pcb_l - cale, epd_y + _epd_scr_t-0.5*buse_w-cale, bx_ep+_epd_tft_b])
        cube([cale+0.5*buse_w, cale, bx_h-2*bx_ep-_epd_tft_b]);
    // derrière
    translate([epd_x + _epd_pcb_l - cale, epd_y + _epd_scr_t+_epd_pcb_t+0.5*buse_w, bx_ep+_epd_tft_b])
        cube([cale+0.5*buse_w, cale, bx_h-2*bx_ep-_epd_tft_b]);
    // droite
    translate([epd_x + _epd_pcb_l + 0.5*buse_w, epd_y + _epd_scr_t - 0.5*buse_w - cale, bx_ep])
        cube([cale, cale + _epd_pcb_t + buse_w, bx_h-2*bx_ep]);
    translate([epd_x + _epd_pcb_l + 0.5*buse_w, epd_y + _epd_scr_t - 0.5*buse_w - cale, plq_z - 0.5*buse_w])
        cube([cale, cale + _epd_pcb_t + buse_w + cale, bx_h-2*bx_ep-plq_z+bx_ep+buse_w/2]);

    // ePD à gauche
    // dessous
    translate([epd_x - 0.5*buse_w-cale, epd_y + _epd_scr_t-0.5*buse_w-cale, bx_ep])
        cube([2*cale - 0.5*buse_w, 2*cale + _epd_pcb_t, _epd_tft_b]);
    // devant
    translate([epd_x - 0.5*buse_w-cale, epd_y + _epd_scr_t-0.5*buse_w-cale, bx_ep])
        cube([2*cale - 0.5*buse_w, cale, bx_h-2*bx_ep]);
    // côté
    translate([epd_x - 0.5*buse_w-cale, epd_y + _epd_scr_t-0.5*buse_w, bx_ep])
        cube([cale, cale + 0.5*buse_w, bx_h-2*bx_ep]);
    // derrière
    translate([epd_x - 0.5*buse_w-cale, epd_y + _epd_scr_t + cale - 0.5*buse_w, bx_ep])
        cube([2*cale - 0.5*buse_w, cale, bx_h-2*bx_ep]);
    translate([epd_x - 0.5*buse_w-cale, epd_y + _epd_scr_t-0.5*buse_w-cale, bx_ep])
        cube([cale, plq_y - epd_y  + cale + buse_w, plq_z -bx_ep - 0.5*buse_w]);
    translate([epd_x - 0.5*buse_w, epd_y + _epd_scr_t + _epd_pcb_t + buse_w, bx_ep])
        cube([cale - 0.5*buse_w, plq_y - epd_y - _epd_pcb_t - 0.5*buse_w, plq_z - bx_ep - 0.5*buse_w]);

    // câle pcb
    translate([epd_x - 0.5*buse_w-cale, epd_y + _epd_scr_t-0.5*buse_w-cale, plq_z - 0.5*buse_w])
        cube([cale, plq_y - epd_y  + cale + buse_w, plq_h + buse_w]);
    translate([epd_x - 0.5*buse_w, epd_y + _epd_scr_t + _epd_pcb_t + buse_w, plq_z - 0.5*buse_w])
        cube([cale - 0.5*buse_w, plq_y - epd_y - _epd_pcb_t - buse_w - cale, plq_h + buse_w]);

    // renfort droit
    hull() {
        translate([epd_x + _epd_pcb_l - cale, epd_y + _epd_scr_t+_epd_pcb_t+0.5*buse_w, plq_z - 0.5*buse_w - cale])
            cube([cale+0.5*buse_w, plq_y - cale - epd_y - _epd_scr_t - _epd_pcb_t - +0.5 * buse_w + 2*cale, cale]);
        translate([bx_l - bx_ep, epd_y + _epd_scr_t+_epd_pcb_t+0.5*buse_w, plq_z - 0.5*buse_w - cale])
            cube([cale, plq_y - cale - epd_y - _epd_scr_t - _epd_pcb_t - +0.5 * buse_w + 2*cale, cale]);
        translate([bx_l - bx_ep, epd_y + _epd_scr_t+_epd_pcb_t+0.5*buse_w, plq_z - 0.5*buse_w - cale - 3*pente*cale])
            cube([cale, plq_y - cale - epd_y - _epd_scr_t - _epd_pcb_t - +0.5 * buse_w + 2*cale, cale]);
    }
    // renfort devant
    hull() {
        translate([epd_x + _epd_pcb_l - cale, epd_y + _epd_scr_t+_epd_pcb_t+0.5*buse_w, plq_z - 0.5*buse_w - cale])
            cube([cale+0.5*buse_w, plq_y - cale - epd_y - _epd_scr_t - _epd_pcb_t - +0.5 * buse_w + 2*cale, cale]);
        translate([epd_x + _epd_pcb_l - cale, epd_y + _epd_scr_t+_epd_pcb_t+0.5*buse_w, plq_z - 0.5*buse_w - cale - 3*pente*cale])
            cube([cale+0.5*buse_w, cale, cale]);
    }
    // câle devant
    translate([epd_x + _epd_pcb_l - cale, epd_y + _epd_scr_t+_epd_pcb_t+0.5*buse_w, plq_z - 0.5*buse_w])
        cube([cale+0.5*buse_w, plq_y - cale - epd_y - _epd_scr_t - _epd_pcb_t - buse_w + cale, plq_h + buse_w]);
    // câle droite
    hull() {
        translate([epd_x + _epd_pcb_l + 0.5*buse_w - cale, plq_y - 0.5*buse_w - cale, plq_z - 0.5*buse_w])
            cube([cale, cale, plq_h + buse_w]);
        translate([epd_x + _epd_pcb_l + 0.5*buse_w + cale/2, plq_y - 0.5*buse_w - cale/2, plq_z - 0.5*buse_w])
            cylinder(r=cale/2, h=plq_h + buse_w, $fn=res);
    }
    hull() {
        translate([epd_x + _epd_pcb_l + 0.5*buse_w + cale/2, plq_y - 0.5*buse_w - cale/2, plq_z - 0.5*buse_w])
            cylinder(r=cale/2, h=plq_h + buse_w, $fn=res);
        translate([epd_x + _epd_pcb_l + 0.5*buse_w*0, plq_y - 0.5*buse_w, plq_z - 0.5*buse_w])
            cube([cale + 0.5*buse_w, cale + 0.5*buse_w, plq_h + buse_w]);
    }
    translate([epd_x + _epd_pcb_l + 0.5*buse_w*0, plq_y , plq_z - 0.5*buse_w])
        cube([bx_l - epd_x - _epd_pcb_l - bx_ep - 0.5*buse_w + 0.5*buse_w, cale, plq_h + buse_w]);


    // câle pcb fond gauche
    hull() {
        translate([plq_x - 0.5*buse_w -cale, bx_w - bx_ep - plq_yoff - cale - 0.5*buse_w, plq_z - 0.5*buse_w - cale])
            cube([4*cale, plq_yoff + cale + 0.5*buse_w + cale*0, cale]);
        translate([plq_x - 0.5*buse_w -cale, bx_w - bx_ep, plq_z - 0.5*buse_w - cale - pente*(plq_yoff+cale)])
            cube([4*cale, cale, cale]);
    }
    // gauche
    translate([plq_x - 0.5*buse_w - cale, bx_w - bx_ep -plq_yoff  - cale - 0.5*buse_w, bx_ep])
        cube([cale, plq_yoff + cale + 0.5*buse_w, plq_z - bx_ep - 0.5*buse_w + plq_h + buse_w]);
    // derrière
    translate([plq_x - 0.5*buse_w - cale, bx_w - bx_ep - plq_yoff- 0.5*buse_w, plq_z- 0.5*buse_w])
        cube([4*cale, plq_yoff+0.5*buse_w, plq_h + buse_w]);

    // câle pcb fond droite
    hull() {
        translate([plq_x + plq_l + 0.5*buse_w - 3*cale, bx_w - bx_ep - plq_yoff - cale - 0.5*buse_w, plq_z - 0.5*buse_w - cale])
            cube([4*cale, plq_yoff + cale + 0.5*buse_w + cale*0, cale]);
        translate([plq_x + plq_l + 0.5*buse_w - 3*cale, bx_w - bx_ep, plq_z - 0.5*buse_w - cale - pente*(plq_yoff+cale)])
            cube([4*cale, cale, cale]);
    }
    // droite
    translate([plq_x + plq_l + 0.5*buse_w*0, bx_w - bx_ep - plq_yoff - cale - 0.5*buse_w, plq_z- 0.5*buse_w])
        cube([cale + 0.5*buse_w, plq_yoff + cale + 0.5*buse_w, plq_h + buse_w]);
    // derrière
    translate([plq_x + plq_l + 0.5*buse_w - 3*cale, bx_w - bx_ep - plq_yoff - 0.5*buse_w, plq_z - 0.5*buse_w])
        cube([4*cale, plq_yoff + 0.5*buse_w, plq_h + buse_w]);

    // transverse ky040
    difference() {
        hull() {
            translate([ky040_x-_ky040_h-_ky040base_h-_ky040axe_h + _ky040ecrou_h + 0.5*buse_w, 0, bx_ep])
                cube([_ky040axe_h-_ky040ecrou_h-buse_w, bx_ep, ky040_z+_ky040axe_x+_ky040axe_r]);
            translate([ky040_x-_ky040_h-_ky040base_h-_ky040axe_h + _ky040ecrou_h + 0.5*buse_w, bx_w-bx_ep, bx_ep])
                cube([_ky040axe_h-_ky040ecrou_h-buse_w, bx_ep, ky040_z+_ky040axe_x+_ky040axe_r]);
            translate([ky040_x-_ky040_h-_ky040base_h-_ky040axe_h + _ky040ecrou_h + 0.5*buse_w, ky040_y+_ky040axe_y, ky040_z+_ky040axe_x])
                rotate([0, 90, 0])
                    cylinder(r=_ky040base_l/2+2*bx_ep, h=_ky040axe_h-_ky040ecrou_h-buse_w, $fn=res);
            }
            // axe
            translate([ky040_x-_ky040_h-_ky040base_h-_ky040axe_h + 0*_ky040ecrou_h, ky040_y+_ky040axe_y, ky040_z+_ky040axe_x])
                rotate([0, 90, 0])
                    cylinder(r=_ky040axe_r+buse_w, h=_ky040axe_h+2*vide, $fn=res);
    }
    // cale ky040
    // devant
    hull() {
        translate([ky040_x-_ky040_h-_ky040base_h-_ky040axe_h + _ky040ecrou_h + 0.5*buse_w, s16340_y+2*s16340_corner_r-cale -0.5*buse_w, bx_ep])
            cube([_ky040axe_h-_ky040ecrou_h-buse_w, cale, ky040_z+_ky040axe_x+_ky040axe_r -bx_ep + cale]);
        translate([ky040_x-2*cale, s16340_y+2*s16340_corner_r-cale -0.5*buse_w, bx_ep])
            cube([s16340_x-buse_w - ky040_x + cale+0.5*buse_w, cale, s16340_b+s16340_h+buse_w]);
    }
    // fond
    hull() {
        translate([ky040_x-_ky040_h-_ky040base_h-_ky040axe_h + _ky040ecrou_h + 0.5*buse_w, s16340_y+s16340_w-2*s16340_corner_r, bx_ep])
            cube([_ky040axe_h-_ky040ecrou_h-buse_w, cale, ky040_z+_ky040axe_x+_ky040axe_r -bx_ep + cale]);
        translate([ky040_x-2*cale, s16340_y+s16340_w-2*s16340_corner_r, bx_ep])
            cube([s16340_x-buse_w - ky040_x + cale+0.5*buse_w, cale, s16340_b+s16340_h+buse_w]);
    }



}

module fond() {
    hull() {
        translate([bx_r, bx_r, 0])
            cylinder(r=bx_r, h=bx_ep, $fn=res);
        translate([bx_l-bx_r, bx_r, 0])
            cylinder(r=bx_r, h=bx_ep, $fn=res);
        translate([bx_l-bx_r, bx_w-bx_r, 0])
            cylinder(r=bx_r, h=bx_ep, $fn=res);
        translate([bx_r, bx_w-bx_r, 0])
            cylinder(r=bx_r, h=bx_ep, $fn=res);
    }
}

module couvercle() {
//color("DarkSalmon") {

nevttop1hx = 3;
nevttop1hy = 5;
nevttop1hw = honeycomb_calc_w(event_bee_r, event_bee_wall, nevttop1hx, nevttop1hy);
nevttop1hh = honeycomb_calc_h(event_bee_r, event_bee_wall, nevttop1hx, nevttop1hy);

nevttop2hx = 3;
nevttop2hy = 3;
nevttop2hw = honeycomb_calc_w(event_bee_r, event_bee_wall, nevttop2hx, nevttop2hy);
nevttop2hh = honeycomb_calc_h(event_bee_r, event_bee_wall, nevttop2hx, nevttop2hy);

    difference() {
        fond();
        translate([bx_ep + nevttop1hh , (bx_w-nevttop1hw)/2, -vide])
            rotate([0, 0, 90])
            honeycomb_cyl(r=event_bee_r, thickness=event_bee_wall, cols=nevttop1hx, h=bx_ep+2*vide, rows=nevttop1hy, center=false, inverse=true);
        translate([bx_l - plq_x - (rcm_plq_x(plq_l, plq_nx, 19) - rcm_plq_x(plq_l, plq_nx, 4))/2 + nevttop2hw/2 , (bx_w-nevttop2hw)/2, -vide])
            rotate([0, 0, 90])
            honeycomb_cyl(r=event_bee_r, thickness=event_bee_wall, cols=nevttop2hx, h=bx_ep+2*vide, rows=nevttop2hy, center=false, inverse=true);
    }

    // Entourage interne
    hull() {
        translate([3*bx_r+buse_w, 3*bx_r+buse_w, bx_ep])
            cylinder(r=bx_r+buse_w/2, h=bx_ep/2, $fn=res);
        translate([bx_l-3*bx_r-buse_w, 3*bx_r+buse_w, bx_ep])
            cylinder(r=bx_r+buse_w/2, h=bx_ep/2, $fn=res);
    }
    hull() {
        translate([bx_l-3*bx_r-buse_w, 3*bx_r+buse_w, bx_ep])
            cylinder(r=bx_r+buse_w/2, h=bx_ep/2, $fn=res);
        translate([bx_l-3*bx_r-buse_w, bx_w-3*bx_r-buse_w, bx_ep])
            cylinder(r=bx_r+buse_w/2, h=bx_ep/2, $fn=res);
    }
    hull() {
        translate([3*bx_r+buse_w, bx_w-3*bx_r-buse_w, bx_ep])
            cylinder(r=bx_r+buse_w/2, h=bx_ep/2, $fn=res);
        translate([3*bx_r+buse_w, 3*bx_r+buse_w, bx_ep])
            cylinder(r=bx_r+buse_w/2, h=bx_ep/2, $fn=res);
    }


plot_l = (event_bee_r-event_bee_wall/2)-buse_w;
plot_h = bx_ep/1;

plot_bck_lx = (bx_l-nevtbckhh)/2 + honeycomb_calc_y(event_bee_r, event_bee_wall, 0, 2) - (event_bee_r-event_bee_wall/2)/2 + buse_w/2;
plot_bck_ly = bx_ep + buse_w/2 + buse_w;
plot_bck_lz = 2*bx_ep + honeycomb_calc_y(event_bee_r, event_bee_wall, 0, 2) - honeycomb_calc_y(event_bee_r, event_bee_wall, 0, 2) + honeycomb_calc_x(event_bee_r, event_bee_wall, 0, 2) - honeycomb_calc_apo(event_bee_r, event_bee_wall) + buse_h;

plot_bck_rx = (bx_l-nevtbckhh)/2 + honeycomb_calc_y(event_bee_r, event_bee_wall, 0, 10) - (event_bee_r-event_bee_wall/2)/2 + buse_w/2;
plot_bck_ry = plot_bck_ly;
plot_bck_rz = plot_bck_lz;

plot_pcb_h = 4.60; // depuis l'épaisseur bx_ep)

    // Plot fond gauche
    hull() {
        translate([plot_bck_lx - plot_l, plot_bck_ly, bx_ep])
            cube([3*plot_l, bx_ep, bx_ep/2]);
        translate([plot_bck_lx, plot_bck_ly, plot_bck_lz + plot_h - bx_ep/2])
            cube([plot_l, bx_ep, bx_ep/2]);
    }
    translate([plot_bck_lx, plot_bck_ly - buse_w/2 - bx_ep/2, plot_bck_lz])
        cube([plot_l, buse_w/2 + bx_ep/2, plot_h]);

    // Plot fond droite
    hull() {
        translate([plot_bck_rx - plot_l, plot_bck_ry, bx_ep])
            cube([3*plot_l, bx_ep, bx_ep/2]);
        translate([plot_bck_rx, plot_bck_ry, plot_bck_rz + plot_h - bx_ep/2])
            cube([plot_l, bx_ep, bx_ep/2]);
    }
    translate([plot_bck_rx, plot_bck_ly - buse_w/2 - bx_ep/2, plot_bck_lz])
        cube([plot_l, buse_w/2 + bx_ep/2, plot_h]);

    // Plot côté gauche
plot_lft_ly = (bx_w-nevtlfthh)/2 + honeycomb_calc_y(event_bee_r, event_bee_wall, 0, 2) - (event_bee_r-event_bee_wall/2)/2 + buse_w/2;
plot_lft_lx = bx_ep + buse_w/2 + buse_w;
plot_lft_lz = 2*bx_ep + honeycomb_calc_y(event_bee_r, event_bee_wall, 0, 2) - honeycomb_calc_y(event_bee_r, event_bee_wall, 0, 2) + honeycomb_calc_x(event_bee_r, event_bee_wall, 0, 2) - honeycomb_calc_apo(event_bee_r, event_bee_wall) + buse_h;

    hull() {
    translate([plot_lft_lx, plot_lft_ly-plot_l, bx_ep])
        cube([bx_ep, 3*plot_l, bx_ep/2]);

    translate([plot_lft_lx, plot_lft_ly, plot_lft_lz + plot_h - bx_ep/2])
        cube([bx_ep, plot_l, bx_ep/2]);
    }
    translate([plot_lft_lx - buse_w/2 - bx_ep/2, plot_lft_ly, plot_lft_lz])
        cube([buse_w/2 + bx_ep/2, plot_l, plot_h]);

    // Plot côté droit
plot_rgt_ly = (bx_w-nevtlfthh)/2 + honeycomb_calc_y(event_bee_r, event_bee_wall, 0, 2) - (event_bee_r-event_bee_wall/2)/2 + buse_w/2;
plot_rgt_lx = bx_l - bx_ep - buse_w/2 - buse_w;
plot_rgt_lz = 2*bx_ep + honeycomb_calc_y(event_bee_r, event_bee_wall, 0, 2) - honeycomb_calc_y(event_bee_r, event_bee_wall, 0, 2) + honeycomb_calc_x(event_bee_r, event_bee_wall, 0, 2) - honeycomb_calc_apo(event_bee_r, event_bee_wall) + buse_h;

    hull() {
    translate([plot_rgt_lx - bx_ep, plot_rgt_ly-plot_l, bx_ep])
        cube([bx_ep, 3*plot_l, bx_ep/2]);

    translate([plot_rgt_lx - bx_ep, plot_rgt_ly, plot_rgt_lz + plot_h - bx_ep/2])
        cube([bx_ep, plot_l, bx_ep/2]);
    }
    translate([plot_rgt_lx, plot_rgt_ly, plot_rgt_lz])
        cube([buse_w/2 + bx_ep/2, plot_l, plot_h]);

    // Plot D32 gauche
    translate([plq_x+rcm_plq_x(plq_l, plq_nx, 4) + cale, bx_w - d32pro_y-_d32pro_w/2, bx_ep])
        cylinder(r1=2*cale, r2=cale, h=bx_h - bx_ep - d32pro_z - _d32pro_h + buse_h, $fn=res);

    // Plot D32 droit
    translate([plq_x+rcm_plq_x(plq_l, plq_nx, 19) - cale, bx_w - d32pro_y-_d32pro_w/2, bx_ep])
        cylinder(r1=2*cale, r2=cale, h=bx_h - bx_ep - d32pro_z - _d32pro_h + buse_h, $fn=res);

//}
}


module plq_epd() {
    color("OrangeRed") {

    difference() {
        union() {
            cube([plq_epd_l, plq_epd_w, plq_epd_t]);
        }

        translate([plq_epd_hole_x, plq_epd_hole_y, -vide])
            cube([plq_epd_hole_l, plq_epd_hole_w, bx_ep+2*vide]);

    }
    }
}


module knob1() {
    difference() {
        union() {
            translate([_ky040rtr_x, _ky040rtr_y, _ky040_h+_ky040base_h+_ky040axe_h])
            color("DarkViolet", alpha=0.75)
            cylinder(r=knob1_r, h=knob1_h, $fn=res);
        }
        difference() {
            translate([_ky040rtr_x, _ky040rtr_y, _ky040_h+_ky040base_h+_ky040axe_h -vide])
                cylinder(r=_ky040rtr_r+1*buse_w/2, h=_ky040rtr_h+2*vide, $fn=res);
            translate([_ky040rtr_x+_ky040rtr_r/2+0*buse_w/4, _ky040rtr_y-_ky040rtr_r-vide, _ky040_h+_ky040base_h+_ky040axe_h -vide+_ky040rtr_h-_ky040rtr_hb+2*vide])
                cube([_ky040rtr_r+vide, 2*_ky040rtr_r+2*vide, _ky040rtr_h+4*vide]);
        }
    }
}

module knob2() {
    
    difference() {
        color("DarkViolet", alpha=0.75) union() {
            hull() {
                translate([_ky040rtr_x-knob1_r+knob2_r, _ky040rtr_y, _ky040_h+_ky040base_h+_ky040axe_h+knob1_h])
                    sphere(r=knob2_r, $fn=res);
                translate([_ky040rtr_x+knob1_r-knob2_r, _ky040rtr_y, _ky040_h+_ky040base_h+_ky040axe_h+knob1_h])
                    sphere(r=knob2_r, $fn=res);
                translate([_ky040rtr_x-knob1_r+knob2_r, _ky040rtr_y, _ky040_h+_ky040base_h+_ky040axe_h+knob1_h+knob2_h-knob2_r])
                    sphere(r=knob2_r, $fn=res);
                translate([_ky040rtr_x+knob1_r-knob2_r, _ky040rtr_y, _ky040_h+_ky040base_h+_ky040axe_h+knob1_h+knob2_h-knob2_r])
                    sphere(r=knob2_r, $fn=res);

            }
            hull() {
                translate([_ky040rtr_x, _ky040rtr_y-knob1_r+knob2_r, _ky040_h+_ky040base_h+_ky040axe_h+knob1_h])
                    sphere(r=knob2_r, $fn=res);
                translate([_ky040rtr_x, _ky040rtr_y+knob1_r-knob2_r, _ky040_h+_ky040base_h+_ky040axe_h+knob1_h])
                    sphere(r=knob2_r, $fn=res);
                translate([_ky040rtr_x, _ky040rtr_y-knob1_r+knob2_r, _ky040_h+_ky040base_h+_ky040axe_h+knob1_h+knob2_h-knob2_r])
                    sphere(r=knob2_r, $fn=res);
                translate([_ky040rtr_x, _ky040rtr_y+knob1_r-knob2_r, _ky040_h+_ky040base_h+_ky040axe_h+knob1_h+knob2_h-knob2_r])
                    sphere(r=knob2_r, $fn=res);
            }
        }
        translate([_ky040rtr_x, _ky040rtr_y, _ky040_h+_ky040base_h+_ky040axe_h+knob1_h-knob1_r-vide])
            cylinder(r=knob1_r+vide, h=knob1_r+vide, $fn=res);
    }

}

if (prt) {

    if (o_box) {
        boite();
    }
    if (o_top) {
        translate([0, -bx_w-bx_ep, 0])
            couvercle();
    }
    if (o_plq_epd) {
        translate([bx_l-plq_epd_l, -bx_w-2*bx_ep-plq_epd_w, 0])
            plq_epd();
    }
    if (o_knob1) {
        translate([0, -bx_w, knob1_h+_ky040_h+_ky040base_h+_ky040axe_h])
            rotate([180, 0, 0])
                knob1();
    }
    if (o_knob2) {
        translate([2.5*knob1_r, -bx_w-2*bx_ep-3*knob1_r, -knob1_h-_ky040_h-_ky040base_h-_ky040axe_h])
                knob2();
    }

} else {

    if (draw_epd) {
        translate([epd_x, epd_y, epd_z])
            %rcm_epd();
    }

    if (o_plq_epd) {
        translate([epd_x, epd_y+_epd_scr_t, bx_h-bx_ep])
            rotate([-90, 0, 0])
                %plq_epd();
    }


    if (draw_plq) {
        translate([plq_x, plq_y, plq_z])
            %rcm_plq(plq_l, plq_w, plq_h, plq_nx, plq_ny);
    }

    if (draw_s8) {
        translate([plq_x+rcm_plq_x(plq_l, plq_nx, 0), plq_y+rcm_plq_y(plq_w, plq_ny, 0), plq_z+plq_h])
            rotate([0, 0, 90])
                %dupontf(2.54, 4, 1, plq_h, dupontf_picot_h, dupontf_body_h);
        translate([plq_x+rcm_plq_x(plq_l, plq_nx, 0), plq_y+rcm_plq_y(plq_w, plq_ny, 0), plq_z+plq_h+dupontf_picot_h+dupontf_body_h+dupontm_picot_h+dupontm_body_h-dupontm_male_h])
            rotate([0, 180, -90])
                %dupontm(2.54, 4, 1, _s8_pcb_h, dupontm_picot_h, dupontm_body_h, dupontm_male_h);

        translate([s8_x, s8_y, s8_z])
            rotate([0, 0, 180])
                %s8();
    }

    if (draw_d32) {
        translate([plq_x+rcm_plq_x(plq_l, plq_nx, 4), plq_y+rcm_plq_y(plq_w, plq_ny, 0), plq_z+plq_h])
            %dupontf(2.54, 16, 1, plq_h, dupontf_picot_h, dupontf_body_h);
        translate([plq_x+rcm_plq_x(plq_l, plq_nx, 19), plq_y+rcm_plq_y(plq_w, plq_ny, 0), plq_z+plq_h+dupontf_picot_h+dupontf_body_h+dupontm_picot_h+dupontm_body_h-dupontm_male_h])
            rotate([0, 180, 0])
                %dupontm(2.54, 16, 1, _d32pro_h, dupontm_picot_h, dupontm_body_h, dupontm_male_h);

        translate([plq_x+rcm_plq_x(plq_l, plq_nx, 4), plq_y+rcm_plq_y(plq_w, plq_ny, 9), plq_z+plq_h])
            %dupontf(2.54, 16, 1, plq_h, dupontf_picot_h, dupontf_body_h);
        translate([plq_x+rcm_plq_x(plq_l, plq_nx, 19), plq_y+rcm_plq_y(plq_w, plq_ny, 9), plq_z+plq_h+dupontf_picot_h+dupontf_body_h+dupontm_picot_h+dupontm_body_h-dupontm_male_h])
            rotate([0, 180, 0])
                %dupontm(2.54, 16, 1, _d32pro_h, dupontm_picot_h, dupontm_body_h, dupontm_male_h);

        translate([d32pro_x, d32pro_y + _d32pro_w, d32pro_z + _d32pro_h])
            rotate([180, 0, 0])
                %d32pro();

    }

    if (draw_ky040) {
        translate([ky040_x, ky040_y, ky040_z])
            rotate([0, 270, 0])
                %ky040(false, false, false, rotor_r, rotor_h);
    }

    if (o_knob1) {
        translate([ky040_x, ky040_y, ky040_z])
            rotate([0, 270, 0])
                %knob1();
    }

    if (o_knob2) {
        translate([ky040_x, ky040_y, ky040_z])
            rotate([0, 270, 0])
                %knob2();
    }

    if (draw_s16340) {
        translate([s16340_x, s16340_y, s16340_z])
            %s16340(false);
    }

    if (o_box) {
        boite();
    }
    if (o_top) {
        translate([0, +bx_w, bx_h])
            rotate([180, 0, 0])
                couvercle();
    }
}
